
/**
 * Timeline View - Cumulative growth chart for selected members
 * Shows: Members joined, Posts, Comments, Likes given, Likes received over time
 */
var MembersTimeline = (function() {
    'use strict';

    let chartInstance = null;
    let cachedRaw = null;
    let cachedMembers = null;
    let cachedContainer = null;
    let currentMode = 'activity'; // 'activity' | 'fetch' | 'all'

    /** Trim raw data based on mode, returns sliced copy */
    function trimData(raw, mode) {
        let startIdx = 0;

        if (mode === 'activity') {
            // First index where any series > 0
            for (let i = 0; i < raw.days.length; i++) {
                if (raw.joined[i] > 0 || raw.posts[i] > 0 || raw.comments[i] > 0 ||
                    raw.likes_given[i] > 0 || raw.likes_received[i] > 0) {
                    startIdx = i;
                    break;
                }
            }
        } else if (mode === 'fetch' && raw.first_fetch_date) {
            // First index >= first_fetch_date
            for (let i = 0; i < raw.days.length; i++) {
                if (raw.days[i] >= raw.first_fetch_date) {
                    startIdx = i;
                    break;
                }
            }
        }
        // mode === 'all' → startIdx stays 0

        return {
            days: raw.days.slice(startIdx),
            joined: raw.joined.slice(startIdx),
            posts: raw.posts.slice(startIdx),
            comments: raw.comments.slice(startIdx),
            likes_given: raw.likes_given.slice(startIdx),
            likes_received: raw.likes_received.slice(startIdx),
        };
    }

    function renderChart(container, raw, members, mode) {
        const data = trimData(raw, mode);

        if (data.days.length === 0) {
            container.innerHTML = '<div style="padding:20px;color:#888;">No timeline data available for the selected members.</div>';
            return;
        }

        // Destroy previous chart instance
        if (chartInstance) {
            chartInstance.destroy();
            chartInstance = null;
        }

        // Determine which range buttons to show
        const hasFetchDate = !!raw.first_fetch_date;

        // Build the DOM
        container.innerHTML = `
            <div style="display:flex; flex-direction:column; height:100%; padding:12px;">
                <div style="display:flex; align-items:center; gap:12px; margin-bottom:8px; flex-shrink:0; flex-wrap:wrap;">
                    <h3 style="margin:0;">Community Timeline (${members.length} members)</h3>
                    <small style="color:#888;">Cumulative growth over time</small>
                    <div style="display:flex; gap:4px; margin-left:auto;" id="timeline-range-toggles"></div>
                </div>
                <div style="display:flex; gap:8px; margin-bottom:8px; flex-shrink:0; flex-wrap:wrap;" id="timeline-toggles"></div>
                <div style="flex:1; min-height:0; position:relative;">
                    <canvas id="timeline-chart"></canvas>
                </div>
                <div style="flex-shrink:0; margin-top:8px; font-size:0.8em; color:#888;" id="timeline-totals"></div>
            </div>
        `;

        // Range toggle buttons
        const rangeContainer = document.getElementById('timeline-range-toggles');
        if (rangeContainer) {
            const modes = [
                { id: 'activity', label: 'Since first activity' },
                { id: 'fetch', label: 'Since first fetch', hidden: !hasFetchDate },
                { id: 'all', label: 'All time' },
            ];
            for (const m of modes) {
                if (m.hidden) continue;
                const btn = document.createElement('button');
                btn.textContent = m.label;
                btn.style.cssText = 'padding:3px 10px; border-radius:4px; border:1px solid #555; background:' +
                    (mode === m.id ? '#444' : 'transparent') + '; color:' +
                    (mode === m.id ? '#fff' : '#888') + '; cursor:pointer; font-size:0.8em;';
                btn.onclick = () => {
                    currentMode = m.id;
                    renderChart(container, raw, members, m.id);
                };
                rangeContainer.appendChild(btn);
            }
        }

        // Totals info
        const lastIdx = data.days.length - 1;
        const totalsEl = document.getElementById('timeline-totals');
        if (totalsEl) {
            totalsEl.innerHTML = `
                Totals &mdash;
                Members: <b>${data.joined[lastIdx]}</b> |
                Posts: <b>${data.posts[lastIdx]}</b> |
                Comments: <b>${data.comments[lastIdx]}</b> |
                Likes given: <b>${data.likes_given[lastIdx]}</b> |
                Likes received: <b>${data.likes_received[lastIdx]}</b> |
                Period: ${data.days[0]} to ${data.days[lastIdx]}
            `;
        }

        // Format labels: show fewer labels for large date ranges
        const labels = data.days.map(d => d.slice(5)); // MM-DD

        const datasets = [
            {
                label: 'Members joined',
                data: data.joined,
                borderColor: '#2ecc71',
                backgroundColor: 'rgba(46, 204, 113, 0.1)',
                tension: 0.3,
                pointRadius: 0,
                fill: false,
                hidden: false,
            },
            {
                label: 'Posts',
                data: data.posts,
                borderColor: '#3498db',
                backgroundColor: 'rgba(52, 152, 219, 0.1)',
                tension: 0.3,
                pointRadius: 0,
                fill: false,
                hidden: false,
            },
            {
                label: 'Comments',
                data: data.comments,
                borderColor: '#e74c3c',
                backgroundColor: 'rgba(231, 76, 60, 0.1)',
                tension: 0.3,
                pointRadius: 0,
                fill: false,
                hidden: false,
            },
            {
                label: 'Likes given',
                data: data.likes_given,
                borderColor: '#f39c12',
                backgroundColor: 'rgba(243, 156, 18, 0.1)',
                tension: 0.3,
                pointRadius: 0,
                fill: false,
                hidden: false,
            },
            {
                label: 'Likes received',
                data: data.likes_received,
                borderColor: '#9b59b6',
                backgroundColor: 'rgba(155, 89, 182, 0.1)',
                tension: 0.3,
                pointRadius: 0,
                fill: false,
                hidden: false,
            },
        ];

        const canvas = document.getElementById('timeline-chart');
        if (!canvas) return;

        chartInstance = new Chart(canvas.getContext('2d'), {
            type: 'line',
            data: { labels, datasets },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    mode: 'index',
                    intersect: false,
                },
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            color: '#ccc',
                            usePointStyle: true,
                            padding: 15,
                        },
                    },
                    tooltip: {
                        callbacks: {
                            title: function(items) {
                                if (items.length > 0) {
                                    return data.days[items[0].dataIndex];
                                }
                                return '';
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        ticks: {
                            color: '#888',
                            maxTicksLimit: 20,
                            maxRotation: 45,
                        },
                        grid: { color: 'rgba(255,255,255,0.05)' },
                    },
                    y: {
                        ticks: { color: '#888' },
                        grid: { color: 'rgba(255,255,255,0.08)' },
                        beginAtZero: true,
                    },
                },
            },
        });
    }

    async function loadAndRender(container, members) {
        container.innerHTML = '<div style="display:flex;align-items:center;justify-content:center;height:100%;color:#888;">Loading timeline...</div>';

        const skoolIds = members.map(m => m.skool_id);
        const res = await post('/api/timeline/members', { skool_ids: skoolIds });

        if (!res.ok) {
            container.innerHTML = '<div style="padding:20px;color:#c00;">Failed to load timeline data</div>';
            return;
        }

        const raw = res.data;

        if (!raw.days || raw.days.length === 0) {
            container.innerHTML = '<div style="padding:20px;color:#888;">No timeline data available for the selected members.</div>';
            return;
        }

        cachedRaw = raw;
        cachedMembers = members;
        cachedContainer = container;

        renderChart(container, raw, members, currentMode);
    }

    return {
        loadAndRender: loadAndRender,
    };
})();
