
/**
 *  Graph Rendering (Vis.js) - Interaktionen zwischen Usern
 */
var MembersConnectionGraph = (function() {
    'use strict';

    let network = null;

    const defaults = {
        gravity: -3000,
        springLength: 250,
        nodeSize: 25,
        fontSize: 12,
        edgeSmooth: 'continuous',
        physicsEnabled: true
    };

    function buildControlsPanel() {
        const slider = (id, label, min, max, value, step) =>
            `<label style="display:flex;align-items:center;gap:6px;white-space:nowrap;">
                ${label}
                <input type="range" id="graph-ctrl-${id}" min="${min}" max="${max}" value="${value}" step="${step || 1}"
                    oninput="document.getElementById('graph-val-${id}').textContent=this.value; MembersConnectionGraph.updateParam('${id}', Number(this.value))">
                <span id="graph-val-${id}" style="min-width:50px;text-align:right;font-weight:bold">${value}</span>
            </label>`;

        return `<div id="graph-controls" style="display:flex;flex-wrap:wrap;gap:10px 20px;padding:6px 10px;
                    border:1px solid #555;border-radius:4px;margin-bottom:4px;background:rgba(0,0,0,0.15);font-size:12px;align-items:center;">
            ${slider('gravity', 'Gravity', -30000, -100, defaults.gravity, 100)}
            ${slider('springLength', 'Spring Length', 30, 800, defaults.springLength, 10)}
            ${slider('nodeSize', 'Node Size', 5, 80, defaults.nodeSize, 1)}
            ${slider('fontSize', 'Font Size', 4, 32, defaults.fontSize, 1)}
            <label style="display:flex;align-items:center;gap:6px;white-space:nowrap;">
                Edge Smooth
                <select id="graph-ctrl-edgeSmooth" onchange="MembersConnectionGraph.updateParam('edgeSmooth', this.value)" style="font-size:12px;">
                    <option value="continuous" ${defaults.edgeSmooth==='continuous'?'selected':''}>continuous</option>
                    <option value="dynamic" ${defaults.edgeSmooth==='dynamic'?'selected':''}>dynamic</option>
                    <option value="cubicBezier" ${defaults.edgeSmooth==='cubicBezier'?'selected':''}>cubicBezier</option>
                    <option value="false">straight</option>
                </select>
            </label>
            <label style="display:flex;align-items:center;gap:6px;white-space:nowrap;">
                Physics
                <input type="checkbox" id="graph-ctrl-physics" ${defaults.physicsEnabled?'checked':''}
                    onchange="MembersConnectionGraph.updateParam('physicsEnabled', this.checked)">
            </label>
            <button onclick="MembersConnectionGraph.restabilize()" style="font-size:12px;padding:2px 8px;">Re-stabilize</button>
        </div>`;
    }

    function updateParam(key, value) {
        if (!network) return;
        switch (key) {
            case 'gravity':
                network.setOptions({ physics: { barnesHut: { gravitationalConstant: value } } });
                break;
            case 'springLength':
                network.setOptions({ physics: { barnesHut: { springLength: value } } });
                break;
            case 'nodeSize':
                network.setOptions({ nodes: { size: value } });
                // update existing nodes
                network.body.data.nodes.forEach(n => {
                    network.body.data.nodes.update({ id: n.id, size: value });
                });
                break;
            case 'fontSize':
                network.setOptions({ nodes: { font: { size: value } } });
                break;
            case 'edgeSmooth':
                if (value === 'false') {
                    network.setOptions({ edges: { smooth: false } });
                } else {
                    network.setOptions({ edges: { smooth: { type: value } } });
                }
                break;
            case 'physicsEnabled':
                network.setOptions({ physics: { enabled: value } });
                break;
        }
    }

    function restabilize() {
        if (!network) return;
        network.stabilize(100);
    }

    /**
     * Lädt und rendert den Interaktions-Graph.
     * Knoten = User mit Profilbild
     * Kanten = Likes (rot) + Comments (blau) zum Post-Autor
     * @param {HTMLElement} container - Container für den Graph
     * @param {Array} filteredMembers - Gefilterte Members (skool_ids werden verwendet)
     */
    async function loadAndRenderInteractionGraph(container, filteredMembers) {
        lib.showLoading('graphing');
        container.innerHTML = '<p style="padding:20px">Lade Graph-Daten...</p>';

        const skoolIds = filteredMembers.map(m => m.skool_id);
        const res = await post('/api/graph/interactions', {
            skool_ids: skoolIds,
            community: currentCommunity
        });
        lib.hideLoading();
        if (!res.ok) {
            container.innerHTML = '<p style="padding:20px;color:red">Fehler beim Laden der Graph-Daten</p>';
            return;
        }

        const { nodes, like_edges, comment_edges } = res.data;

        if (nodes.length === 0) {
            container.innerHTML = '<p style="padding:20px">Keine User gefunden</p>';
            return;
        }

        const hideVal = await ConfigEntry.get("hide_profile_images");
        const hideImages = hideVal === "1";

        const visNodes = nodes.map(n => {
            const node = {
                id: n.id,
                label: n.name,
                size: defaults.nodeSize,
                title: `@${n.name}\nRole: ${n.role}`,
                group: n.role,
                borderWidth: 2
            };
            if (hideImages || !n.id) {
                node.shape = 'dot';
            } else {
                node.shape = 'circularImage';
                node.image = `/api/image/${n.id}`;
            }
            return node;
        });

        const visEdges = [
            ...like_edges.map(e => ({
                from: e.source,
                to: e.target,
                value: e.weight,
                color: { color: '#e74c3c', highlight: '#c0392b' },
                title: `${e.weight} Like${e.weight > 1 ? 's' : ''}`,
                arrows: 'to'
            })),
            ...comment_edges.map(e => ({
                from: e.source,
                to: e.target,
                value: e.weight,
                color: { color: '#3498db', highlight: '#2980b9' },
                title: `${e.weight} Comment${e.weight > 1 ? 's' : ''}`,
                arrows: 'to'
            }))
        ];

        // Controls + Stats + Graph
        container.innerHTML = buildControlsPanel();

        const stats = document.createElement('small');
        stats.style.cssText = 'position:absolute;top:5px;left:10px;background:rgba(255,255,255,0.9);padding:4px 8px;border-radius:4px;z-index:10';
        stats.textContent = `${nodes.length} Users | ${like_edges.length} Like-Verbindungen (rot) | ${comment_edges.length} Comment-Verbindungen (blau)`;

        const graphWrap = document.createElement('div');
        graphWrap.style.cssText = 'width:100%;flex:1;position:relative;min-height:0';
        graphWrap.appendChild(stats);

        const graphDiv = document.createElement('div');
        graphDiv.style.cssText = 'width:100%;height:100%';
        graphWrap.appendChild(graphDiv);

        container.appendChild(graphWrap);

        network = new vis.Network(graphDiv, { nodes: visNodes, edges: visEdges }, {
            nodes: {
                font: { size: defaults.fontSize },
                scaling: { min: 20, max: 40 }
            },
            edges: {
                smooth: { type: defaults.edgeSmooth },
                scaling: { min: 1, max: 8 }
            },
            physics: {
                stabilization: { iterations: 50, fit: true },
                barnesHut: {
                    gravitationalConstant: defaults.gravity,
                    springLength: defaults.springLength
                }
            },
            groups: {
                admin: { color: { border: '#e74c3c', background: '#fadbd8' } },
                moderator: { color: { border: '#f39c12', background: '#fef5e7' } },
                member: { color: { border: '#3498db', background: '#ebf5fb' } }
            }
        });
    }

    return {
        loadAndRenderInteractionGraph: loadAndRenderInteractionGraph,
        updateParam: updateParam,
        restabilize: restabilize
    };
}());